/** @file   objective.cpp
 * @brief   Implementation of Objective - class
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#include "objective.h"
#include "warglobals.h"
#include "utils.h"
#include "SoundSamples.h"
using namespace eng2d;
using std::vector;
using std::string;
using std::istream;

namespace WeWantWar {

/** Constructor
 */
Objective::Objective() :
  m_countdownFlag( false ),
  m_timer( -1 ),
  m_timermode( M_FAILS ),
  
  m_playerInsideFlag( false ),
  m_playerRect(),

  m_areaClearFlag( false ),
  m_areaClearRects(),
  m_areaClearID( 0 ),
  m_areaClearMode( M_FAILS ),
  
  m_killedIDFlag( false ),
  m_killedIDCodes(),
  m_killedIDMode( M_FAILS ),
  
  m_inactiveTimer( 5 ),
  m_accomplishedSound( -1 )
{
}



/** Destructor
 */
Objective::~Objective()
{
  for ( int i=0; i < m_events.size(); i++ ) {
    delete m_events.at(i);
    m_events.at(i) = 0;
  }
}



/** updates this objective
 */
Objective::State Objective::update()
{
  Player* pPlayer = WarGlobals::pObjTable->pPlayer;
  
  
  // Update the events and timers
  for ( int i=0; i < m_events.size(); i++ ) {
    m_events.at(i)->update();
  }
  
  m_inactiveTimer -= 1;
  if ( m_inactiveTimer > 0 ) {
    return INCOMPLETE;
  }
  
  State ret = INCOMPLETE;
  
  // First we check the timer-condition.
  if ( m_countdownFlag == true ) {
    m_timer -= 1;
    if ( m_timer < 0 ) {
      if ( m_timermode == M_SUCCESSFUL ) {
        m_state = ACCOMPLISHED;
        ret = ACCOMPLISHED;
      } else {
        m_state = FAILED;
        ret = FAILED;
      }
    }
    
    // make sound
    int soundIndex = -1;
    switch (m_timer) {
      case 40:
      case 120:
      case 200:
      case 280:
        soundIndex = SMP_CLOCK1;
        break;
      case 0:
      case 80:
      case 160:
      case 240:
        soundIndex = SMP_CLOCK2;
        break;
      default:
        soundIndex = -1;
        break;
    }
    if ( soundIndex != -1 ) {
      Sound::playSample( soundIndex, false );
    }
  }

  // Then the player inside - condition.
  if ( m_playerInsideFlag == true ) {
    if ( m_playerRect.pointInside( pPlayer->position() ) == true ) {
      m_state = ACCOMPLISHED;
      ret = ACCOMPLISHED;
    }
  }
  
  // Are there enemies inside the rectangles.
  if ( m_areaClearFlag == true ) {
    bool areaClear = true;
    vector<GameObject*> fooVec;
    WarGlobals::pObjTable->getObjectsWithID( m_areaClearID, fooVec );
    for ( int i=0; i < fooVec.size(); i++ ) {
      if ( fooVec.at(i)->state() != GameObject::STATE_KILLED ) {
        // We care only about living objects.
        for ( int j=0; j < m_areaClearRects.size(); j++ ) {
          if ( m_areaClearRects.at(j).pointInside( fooVec.at(i)->position() ) == true ) {
            areaClear = false;
          }
        }
      }
    }
    
    if ( areaClear == true ) {
      if ( m_areaClearMode == M_FAILS ) {
        m_state = FAILED;
      } else {
        m_state = ACCOMPLISHED;
      }
      ret = m_state;
    }
  }
  
  // And are there objects with certain ID-codes still alive.
  if ( m_killedIDFlag == true ) {
    bool aliveObjects = false;
    for ( int i=0; i < m_killedIDCodes.size(); i++ ) {
      vector<GameObject*> fooVec;
      WarGlobals::pObjTable->getObjectsWithID( m_killedIDCodes.at(i), fooVec );
      for ( int k=0; k < fooVec.size(); k++ ) {
        if ( fooVec.at(k)->state() != GameObject::STATE_KILLED ) {
          aliveObjects = true;
          break;
        }
      }
    }
    if ( aliveObjects == false ) {
      if ( m_killedIDMode == M_FAILS ) {
        m_state = FAILED;
      } else {
        m_state = ACCOMPLISHED;
      }
      ret = m_state;
    }
  }
  return ret;
}



/** Reads the data from given stream
 */
int Objective::readData( std::istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    std::string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "</objective>" ) {
      break;
      
    } else if ( tmp == "<event>" ) {
      DynamicEvent* pE = new DynamicEvent();
      if ( pE->readData( rIn ) != 0 ) {
        return -1;
      }
      this->addEvent( pE );
      
    } else if ( tmp == "<message>" ) {
      if ( this->readMessage( rIn ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<finishing_conditions>" ) {
      if ( this->readFinishingConditions( rIn ) != 0 ) {
        return -1;
      }
      
    }
  }
  return 0;
}


/** Returns the current state
 */
Objective::State Objective::state() const
{
  return m_state;
}




/** Returns the timer
 */
int Objective::getTimer() const
{
  return m_timer;
}



/** Returns the message
 */
const std::string& Objective::getMessage() const
{
  return m_message;
}



/** Returns the soundnumber
 */
int Objective::soundNumber() const
{
  return m_accomplishedSound;
}


/** Adds new event to this objective
 */
void Objective::addEvent( DynamicEvent* pEvent )
{
  m_events.push_back( pEvent );
}



/** Reads the message
 */
int Objective::readMessage( istream& rIn )
{
  m_message = "";
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    string tmp;
    rIn >> tmp;
    if ( tmp == "</message>" ) {
      return 0;
    }
    
    m_message += tmp;
    m_message += " ";
  }
}



/** Reads the finishing conditions.
 */
int Objective::readFinishingConditions( istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );

    } else if ( tmp == "<countdown>" ) {
      if ( this->readCountdown( rIn ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<player_inside>" ) {
      if ( this->readPlayerInside( rIn ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<area_clear>" ) {
      if ( this->readAreaClear( rIn ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<killed_id>" ) {
      if ( this->readKilledID( rIn ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "</finishing_conditions>" ) {
      return 0;
      
    }
  }
}



/** Reads the contents of COUNTDOWN - element
 */
int Objective::readCountdown( istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "status:" ) {
      // Read the status-string and flag on or off.
      rIn >> tmp;
      if ( tmp == "on" ) {
        m_countdownFlag = true;
      } else {
        m_countdownFlag = false;
      }
      
    } else if ( tmp == "from:" ) {
      // read the timervalue from where we start the countdown.
      rIn >> m_timer;
      
    } else if ( tmp == "mode:" ) {
      // Read the mode.
      rIn >> tmp;
      m_timermode = M_FAILS;
      if ( tmp == "accomplished" ) {
        m_timermode = M_SUCCESSFUL;
      }
      
    } else if ( tmp == "</countdown>" ) {
      return 0;
      
    } else {
      int ret = alert( "UNKNOWN PROPERTY!",
                       "COUNTDOWN - element", tmp.c_str(),
                       "continue", "abort", 0,0 );
      if ( ret == 2 ) {
        return -1;
      }
    }
  }
}



/** Reads the contents of PLAYER_INSIDE - element
 */
int Objective::readPlayerInside( istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "</player_inside>" ) {
      return 0;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "status:" ) {
      rIn >> tmp;
      if ( tmp == "on" ) {
        m_playerInsideFlag = true;
      } else {
        m_playerInsideFlag = false;
      }
      
    } else if ( tmp == "rect:" ) {
      int x1;
      int y1;
      int x2;
      int y2;
      rIn >> x1 >> y1 >> x2 >> y2;
      m_playerRect = Rect2D( Vec2D(x1*32, y1*32), Vec2D(x2*32, y2*32) );
      
    } else {
      int ret = alert( "UNKNOWN PROPERTY!",
                       "PLAYER_INSIDE - element", tmp.c_str(),
                       "continue", "abort", 0,0 );
      if ( ret == 2 ) {
        return -1;
      }
    }
  }
}



/** Reads the contents of ENEMIES_INSIDE - element
 */
int Objective::readAreaClear( istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    string tmp;
    rIn >> tmp;
    if ( tmp == "</area_clear>" ) {
      return 0;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "status:" ) {
      rIn >> tmp;
      m_areaClearFlag = false;
      if ( tmp == "on" ) {
        m_areaClearFlag = true;
      }
      
    } else if ( tmp == "id:" ) {
      rIn >> m_areaClearID;
      
    } else if ( tmp == "rect:" ) {
      int x1, y1, x2, y2;
      rIn >> x1 >> y1 >> x2 >> y2;
      Rect2D rect( Vec2D(x1*32, y1*32), Vec2D(x2*32, y2*32) );
      m_areaClearRects.push_back( rect );
      
    } else if ( tmp == "mode:" ) {
      rIn >> tmp;
      m_areaClearMode = M_FAILS;
      if ( tmp == "accomplished" ) {
        m_areaClearMode = M_SUCCESSFUL;
      }
      
    } else {
      int ret = alert( "UNKNOWN PROPERTY!",
                       "ENEMIES_INSIDE - element", tmp.c_str(),
                       "continue", "abort", 0,0 );
      if ( ret == 2 ) {
        return -1;
      }
    }
  }
}



/** Reads the OBJECTS_KILLED - element
 */
int Objective::readKilledID( istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    string tmp;
    rIn >> tmp;
    if ( tmp == "</killed_id>" ) {
      return 0;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "status:" ) {
      rIn >> tmp;
      m_killedIDFlag = false;
      if ( tmp == "on" ) {
        m_killedIDFlag = true;
      }
      
    } else if ( tmp == "id:" ) {
      GameObject::IDCode id;
      rIn >> id;
      m_killedIDCodes.push_back( id );
      
    } else if ( tmp == "mode:" ) {
      rIn >> tmp;
      m_killedIDMode = M_FAILS;
      if ( tmp == "accomplished" ) {
        m_killedIDMode = M_SUCCESSFUL;
      }
      
    } else {
      int ret = alert( "UNKNOWN PROPERTY!",
                       "OBJECTS_KILLED - element", tmp.c_str(),
                       "continue", "abort", 0,0 );
      if ( ret == 2 ) {
        return -1;
      }
    }
  }
}


} // end of namespace
